<?php
session_start();
require_once "../formulario/config.php";

// Seguridad: Chequea admin y tiempo de sesión (30 min)
requireAdmin();
if (isset($_SESSION['admin_login_time']) && (time() - $_SESSION['admin_login_time'] > 1800)) {
    session_destroy();
    header("Location: login.php?tiempo_expirado=1");
    exit();
}

// Procesa aprobación/rechazo con código secreto (CSRF)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['accion'], $_POST['id'], $_POST['csrf'])) {
    if (!validateCSRF($_POST['csrf'])) {
        error_log("Intento de truco CSRF desde " . $_SERVER['REMOTE_ADDR']);
        header("Location: panel_admin.php?truco=1");
        exit();
    }
    $accion = filter_var($_POST['accion'], FILTER_SANITIZE_STRING);
    $id = filter_var($_POST['id'], FILTER_VALIDATE_INT);
    if (!$id || !in_array($accion, ['aprobado', 'rechazado'])) {
        header("Location: panel_admin.php?invalido=1");
        exit();
    }

    if ($accion === 'aprobado') {
        // Obtener datos del club del estudiante
        $sqlClub = "SELECT c.nombre, c.limite, c.inscritos, c.tipo 
                    FROM estudiantes e 
                    JOIN clubes c ON e.club = c.nombre
                    WHERE e.id = ?";
        $stmtClub = $conn->prepare($sqlClub);
        $stmtClub->bind_param("i", $id);
        $stmtClub->execute();
        $clubData = $stmtClub->get_result()->fetch_assoc();
        $stmtClub->close();

        if (!$clubData) {
            header("Location: panel_admin.php?club_no_encontrado=1");
            exit();
        }

        $limite = $clubData['limite'];
        $inscritos = $clubData['inscritos'];
        $tipo = $clubData['tipo'];

        $puedeAprobar = false;
        if ($tipo === 'flexible' || $tipo === 'gratis') {
            $puedeAprobar = true;
        } elseif ($tipo === 'con_limite' && $inscritos < $limite) {
            $puedeAprobar = true;
        }

        if ($puedeAprobar) {
            // Actualizar estado del estudiante
            $sql = "UPDATE estudiantes SET estado = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("si", $accion, $id);
            $stmt->execute();
            $stmt->close();

            // Si tiene límite, incrementar inscritos
            if ($tipo === 'con_limite') {
                $sqlUpd = "UPDATE clubes SET inscritos = inscritos + 1 WHERE nombre = ?";
                $stmtUpd = $conn->prepare($sqlUpd);
                $stmtUpd->bind_param("s", $clubData['nombre']);
                $stmtUpd->execute();
                $stmtUpd->close();
            }
        } else {
            header("Location: panel_admin.php?cupo_lleno=1");
            exit();
        }

    } elseif ($accion === 'rechazado') {
        // Rechazar estudiante
        $sql = "UPDATE estudiantes SET estado = ?, comprobante = NULL WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("si", $accion, $id);
        $stmt->execute();
        $stmt->close();
    }

    header("Location: panel_admin.php");
    exit();
}

// Mensajes de error
$error_msg = '';
if (isset($_GET['truco'])) $error_msg = "Solicitud inválida. Intenta de nuevo.";
if (isset($_GET['invalido'])) $error_msg = "Acción no válida.";
if (isset($_GET['cupo_lleno'])) $error_msg = "No se puede aprobar: cupo del club lleno.";
if (isset($_GET['club_no_encontrado'])) $error_msg = "No se encontró el club del estudiante.";

// Función para mostrar tabla (igual que tu versión)
function mostrarTabla($conn, $estado)
{
    $stmt = $conn->prepare("SELECT id, fecha_inscripcion, apellidos, nombres, curso, codigo_credencial, celular, club, metodo_pago, comprobante, estado 
                            FROM estudiantes WHERE estado = ? ORDER BY fecha_inscripcion DESC");
    $stmt->bind_param("s", $estado);
    $stmt->execute();
    $result = $stmt->get_result();

    $exportId = ($estado === 'aprobado') ? 'id="export-aprobado"' : '';

    echo "<div class='export-section'>";
    echo "<a href='export_estados.php?estado=" . $estado . "' class='export-btn' target='_blank' $exportId>📊 Exportar Excel</a>";
    echo "</div>";

    echo "<table class='tabla-admin'>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Fecha y Hora Inscripción</th>
                    <th>Apellidos</th>
                    <th>Nombres</th>
                    <th>Curso</th>
                    <th>Código Credencial</th>
                    <th>Celular</th>
                    <th>Club</th>
                    <th>Método de Pago</th>
                    <th>Comprobante</th>
                    <th>Estado</th>";

    if ($estado === 'pendiente') echo "<th>Acción</th>";

    echo "</tr></thead><tbody>";

    while ($row = $result->fetch_assoc()) {
        echo "<tr data-club='" . htmlspecialchars($row['club']) . "'>";
        echo "<td>" . htmlspecialchars($row['id']) . "</td>";
        echo "<td>" . (isset($row['fecha_inscripcion']) ? date('d/m/Y H:i:s', strtotime($row['fecha_inscripcion'])) : 'N/A') . "</td>";
        echo "<td>" . htmlspecialchars($row['apellidos']) . "</td>";
        echo "<td>" . htmlspecialchars($row['nombres']) . "</td>";
        echo "<td>" . htmlspecialchars($row['curso']) . "</td>";
        echo "<td>" . htmlspecialchars($row['codigo_credencial']) . "</td>";
        echo "<td>" . (decryptData($row['celular']) ?? $row['celular']) . "</td>";
        echo "<td>" . htmlspecialchars($row['club']) . "</td>";
        echo "<td>" . htmlspecialchars($row['metodo_pago']) . "</td>";

        if (!empty($row['comprobante'])) {
            echo "<td><a href='../uploads/" . urlencode($row['comprobante']) . "' target='_blank'>Ver PDF</a></td>";
        } else {
            echo "<td>No subido</td>";
        }

        echo "<td>" . htmlspecialchars($row['estado']) . "</td>";

        if ($estado === 'pendiente') {
            $csrf = generateCSRF();
            echo "<td>
                    <form method='POST' style='display:inline;'>
                        <input type='hidden' name='csrf' value='" . $csrf . "'>
                        <input type='hidden' name='id' value='" . $row['id'] . "'>
                        <button type='submit' name='accion' value='aprobado' class='btn-aprobar'>✅</button>
                    </form>
                    <form method='POST' style='display:inline;'>
                        <input type='hidden' name='csrf' value='" . $csrf . "'>
                        <input type='hidden' name='id' value='" . $row['id'] . "'>
                        <button type='submit' name='accion' value='rechazado' class='btn-rechazar'>❌</button>
                    </form>
                  </td>";
        }

        echo "</tr>";
    }

    echo "</tbody></table>";
    $stmt->close();
}
?>

    <!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<title>Panel de Administración</title>
<link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
    }
    body {
        font-family: "Poppins", sans-serif;
        background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        color: #333;
        line-height: 1.6;
    }

    /* Sidebar */
    .sidebar {
        width: 260px;
        background: linear-gradient(180deg, #0e1a44 0%, #b30089 100%);
        color: white;
        height: 100vh;
        padding: 30px 20px;
        position: fixed;
        left: 0;
        top: 0;
        box-shadow: 4px 0 20px rgba(0,0,0,0.1);
        overflow-y: auto;
    }
    .sidebar h2 {
        font-size: 24px;
        margin-bottom: 30px;
        text-align: center;
        font-weight: 600;
    }
    .sidebar h3 {
        font-size: 16px;
        margin: 20px 0 10px 0;
        font-weight: 500;
        color: rgba(255,255,255,0.9);
    }
    .sidebar button {
        background: transparent;
        border: none;
        color: rgba(255,255,255,0.9);
        text-align: left;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        cursor: pointer;
        border-radius: 12px;
        transition: all 0.3s ease;
        margin-bottom: 8px;
        font-weight: 400;
    }
    .sidebar button:hover {
        background: rgba(255,255,255,0.15);
        color: white;
        transform: translateX(4px);
    }
    .sidebar button.active {
        background: rgba(255,255,255,0.25);
        color: white;
        font-weight: 600;
    }
    .sidebar hr {
        border: none;
        height: 1px;
        background: rgba(255,255,255,0.2);
        margin: 20px 0;
    }
    .sidebar a {
        display: block;
        color: rgba(255,255,255,0.8);
        text-decoration: none;
        padding: 12px 16px;
        border-radius: 12px;
        transition: all 0.3s ease;
        margin-top: 10px;
    }
    .sidebar a:hover {
        background: rgba(255,255,255,0.15);
        color: white;
    }

    /* Contenedor principal */
    .main {
        margin-left: 260px;
        padding: 40px;
        width: calc(100% - 260px);
        min-height: 100vh;
    }

    /* Header principal */
    h1 {
        color: #0e1a44;
        font-size: 28px;
        margin-bottom: 30px;
        font-weight: 700;
        text-align: center;
        background: linear-gradient(135deg, #0e1a44, #b30089);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
    }

    /* Buscador */
    .buscador-section {
        text-align: center;
        margin-bottom: 30px;
    }
    .buscador {
        width: 100%;
        max-width: 500px;
        padding: 14px 20px;
        font-size: 14px;
        border-radius: 50px;
        border: 2px solid #ddd;
        outline: none;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }
    .buscador:focus {
        border-color: #b30089;
        box-shadow: 0 0 20px rgba(179, 0, 137, 0.2);
    }
    .buscador::placeholder {
        color: #999;
    }

    /* Export Section */
    .export-section {
        text-align: center;
        margin-bottom: 20px;
    }
    .export-btn {
        display: inline-block;
        background: linear-gradient(135deg, #28a745, #20c997);
        color: white;
        text-decoration: none;
        padding: 10px 20px;
        border-radius: 25px;
        font-weight: 500;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
        font-size: 14px;
    }
    .export-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
        color: white;
    }

    /* Tablas */
    .tabla-contenedor {
        background: white;
        border-radius: 20px;
        padding: 30px;
        box-shadow: 0 10px 40px rgba(0,0,0,0.1);
        margin-bottom: 40px;
        overflow: hidden;
    }
    .tabla-contenedor h2 {
        color: #0e1a44;
        font-size: 22px;
        margin-bottom: 20px;
        font-weight: 600;
        text-align: center;
        background: linear-gradient(135deg, #0e1a44, #b30089);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
    }
    table {
        width: 100%;
        border-collapse: collapse;
        font-size: 13px;
    }
    th, td {
        padding: 12px 8px;
        border-bottom: 1px solid #eee;
        text-align: left;
    }
    th {
        background: linear-gradient(135deg, #0e1a44, #b30089);
        color: white;
        font-weight: 500;
        font-size: 12px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    tr {
        transition: all 0.3s ease;
    }
    tr:hover {
        background: linear-gradient(135deg, #f8f9ff, #fdf2ff);
        transform: scale(1.01);
        box-shadow: 0 4px 15px rgba(0,0,0,0.05);
    }
    .btn-aprobar {
        background: linear-gradient(135deg, #00c851, #00b140);
        color: white;
        border: none;
        border-radius: 6px;
        padding: 6px 12px;
        cursor: pointer;
        font-size: 12px;
        transition: all 0.3s ease;
        box-shadow: 0 2px 8px rgba(0,200,81,0.3);
    }
    .btn-aprobar:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0,200,81,0.4);
    }
    .btn-rechazar {
        background: linear-gradient(135deg, #ff4444, #cc0000);
        color: white;
        border: none;
        border-radius: 6px;
        padding: 6px 12px;
        cursor: pointer;
        font-size: 12px;
        transition: all 0.3s ease;
        box-shadow: 0 2px 8px rgba(255,68,68,0.3);
    }
    .btn-rechazar:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(255,68,68,0.4);
    }

    /* Responsive */
    @media (max-width: 768px) {
        .sidebar {
            width: 100%;
            height: auto;
            position: relative;
        }
        .main {
            margin-left: 0;
            width: 100%;
            padding: 20px;
        }
        .tabla-contenedor {
            padding: 20px;
        }
        table {
            font-size: 11px;
        }
        th, td {
            padding: 8px 4px;
        }
        .buscador {
            max-width: 100%;
        }
    }
</style>
</head>
<body>
    <div class="sidebar">
        <h2>📊 Panel Admin</h2>
        <button class="active" onclick="mostrarSeccion('pendiente')">🕒 Pendientes</button>
        <button onclick="mostrarSeccion('aprobado')">✅ Aprobados</button>
        <button onclick="mostrarSeccion('rechazado')">❌ Rechazados</button>
        <hr>
        <h3>🎓 Clubes</h3>
        <?php
        $clubs = ["Voley","Futsal","Basquetbol","Fisica","Comunicacion","Conquistadores","Poesia y Oratoria","Teologia","Medicina","Quimica","Dactilografia"];
        foreach ($clubs as $club) {
            echo "<button onclick=\"filtrarClub('$club')\">$club</button>";
        }
        ?>
        <hr>
        <a href="logout.php">🚪 Cerrar sesión</a>
    </div>

    <div class="main">
        <h1>Panel de Administración</h1>
        <div class="buscador-section">
            <input type="text" id="buscador" class="buscador" placeholder="Buscar en todas las tablas..." onkeyup="buscarGlobal()">
        </div>

        <div id="pendiente" class="tabla-contenedor">
            <h2>🕒 Pendientes</h2>
            <?php mostrarTabla($conn, 'pendiente'); ?>
        </div>

        <div id="aprobado" class="tabla-contenedor" style="display:none;">
            <h2 id="titulo-aprobado">✅ Aprobados</h2>
            <?php mostrarTabla($conn, 'aprobado'); ?>
        </div>

        <div id="rechazado" class="tabla-contenedor" style="display:none;">
            <h2>❌ Rechazados</h2>
            <?php mostrarTabla($conn, 'rechazado'); ?>
        </div>
    </div>

<script>
let currentActiveBtn = null;
let currentClubFilter = null;

function mostrarSeccion(id) {
    // Ocultar todas las secciones
    document.querySelectorAll('.tabla-contenedor').forEach(div => div.style.display = 'none');
    
    // Mostrar la sección seleccionada
    document.getElementById(id).style.display = 'block';
    
    // Actualizar botones activos
    document.querySelectorAll('.sidebar button').forEach(btn => btn.classList.remove('active'));
    currentActiveBtn = event.target;
    if (currentActiveBtn) {
        currentActiveBtn.classList.add('active');
    }
    
    // Resetear título de aprobados y limpiar filtro si se sale del apartado
    if (id === 'aprobado') {
        document.getElementById('titulo-aprobado').textContent = '✅ Aprobados';
        if (currentClubFilter) {
            limpiarFiltroClub();
            currentClubFilter = null;
        }
        document.getElementById('export-aprobado').href = 'export_estados.php?estado=aprobado';
    }
    
    // Limpiar filtros de club
    if (id !== 'aprobado') {
        limpiarFiltroClub();
    }
    buscarGlobal(); // Refrescar búsqueda
}

function filtrarClub(club) {
    mostrarSeccion('aprobado');
    
    // Actualizar título
    document.getElementById('titulo-aprobado').textContent = `✅ Aprobados de ${club}`;
    
    // Filtrar filas
    const filas = document.querySelectorAll('#aprobado tbody tr');
    let visibleCount = 0;
    filas.forEach(f => {
        const clubCelda = f.getAttribute('data-club');
        if (clubCelda === club) {
            f.style.display = '';
            visibleCount++;
        } else {
            f.style.display = 'none';
        }
    });
    
    // Actualizar enlace de export para incluir el club
    document.getElementById('export-aprobado').href = 'export_estados.php?estado=aprobado&club=' + encodeURIComponent(club);
    
    currentClubFilter = club;
    
    // Actualizar botón activo (mantener el de aprobados)
    document.querySelectorAll('.sidebar button').forEach(btn => btn.classList.remove('active'));
    const btnAprobados = Array.from(document.querySelectorAll('.sidebar button')).find(btn => btn.textContent.includes('Aprobados'));
    if (btnAprobados) btnAprobados.classList.add('active');
}

function limpiarFiltroClub() {
    const filas = document.querySelectorAll('#aprobado tbody tr');
    filas.forEach(f => f.style.display = '');
    if (currentClubFilter) {
        document.getElementById('titulo-aprobado').textContent = '✅ Aprobados';
        document.getElementById('export-aprobado').href = 'export_estados.php?estado=aprobado';
        currentClubFilter = null;
    }
}

function buscarGlobal() {
    const filtro = document.getElementById('buscador').value.toLowerCase();
    document.querySelectorAll('tbody tr').forEach(row => {
        const texto = row.innerText.toLowerCase();
        row.style.display = texto.includes(filtro) ? '' : 'none';
    });
}

// Inicializar
document.addEventListener('DOMContentLoaded', function() {
    mostrarSeccion('pendiente'); // Mostrar pendientes por defecto
});
</script>
</body>
</html>


